/*******************************************************************/
/*                                                                 */
/*                      ADOBE CONFIDENTIAL                         */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                     */
/*                                                                 */
/* Copyright 2002 Adobe Systems Incorporated                       */
/* All Rights Reserved.                                            */
/*                                                                 */
/* NOTICE:  All information contained herein is, and remains the   */
/* property of Adobe Systems Incorporated and its suppliers, if    */
/* any.  The intellectual and technical concepts contained         */
/* herein are proprietary to Adobe Systems Incorporated and its    */
/* suppliers and may be covered by U.S. and Foreign Patents,       */
/* patents in process, and are protected by trade secret or        */
/* copyright law.  Dissemination of this information or            */
/* reproduction of this material is strictly forbidden unless      */
/* prior written permission is obtained from Adobe Systems         */
/* Incorporated.                                                   */
/*                                                                 */
/*******************************************************************/

/*
**-----------------------------------------------------------------------------
** Effect File Variables
**-----------------------------------------------------------------------------
*/

uniform float4x4	gWorldViewProj : WorldViewProjection; // This matrix will be loaded by the application
uniform float4x4	gWorldViewInverse;
uniform float4x4	gWorld;
uniform float4		gLightVector;
uniform float4		gEyePosition;
uniform float		gBumpAmount;
uniform float		gCentrePointX;
uniform float		gCentrePointY;
uniform float		gFleckAmount;
uniform float		gReflectivity;
uniform float		gCurrentParameter;
uniform float		gAspectRatio;
uniform float		gPAR;

uniform float2		gModelOffset;
uniform float		gModelScaling;
uniform float2		gUVOffset;
uniform float		gUVScaling;

textureCUBE			gEnvTexture;
texture				gFleckTexture;
texture				gVideoTexture;
texture				gBumpTexture; 



/*
**-----------------------------------------
**		Sampler States
**-----------------------------------------
*/
//incoming video texture
sampler Sampler = sampler_state
{
    Texture   = (gVideoTexture);
    MipFilter = LINEAR;
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};


//environment texture loaded to show effect of room
samplerCUBE SamplerEnv = sampler_state
{
    Texture   = (gEnvTexture);
    MipFilter = LINEAR;
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};

//normal texture to show noise/flecks
sampler SamplerFleck = sampler_state
{
    Texture   = (gFleckTexture);
    MipFilter = LINEAR;
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};

sampler SamplerBump = sampler_state
{
    Texture   = (gBumpTexture);
    MipFilter = LINEAR;
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};



/*
**-----------------------------------------------------------------------------
** Vertex Definitions
**-----------------------------------------------------------------------------
** APP_OUTPUT is the structure in which we receive the vertices from the application
*/
struct APP_OUTPUT
{
    float3 mPosition	: POSITION;
	float3 mNormal		: NORMAL;
	float2 mTexCoord	: TEXCOORD0;
	
};

/* 
** Pixel Shader structure declaration
*/

struct VS_OUTPUT 
{
  float4 mHPosition		: POSITION;		// coord position in window
  float2 mTexcoord		: TEXCOORD0;	// wavy or fleck map texture coordinates
  float3 mLightVec		: TEXCOORD1;		// position of light relative to point
  float3 mHalfVec		: TEXCOORD2;		// Blinn halfangle
  float3 mIncidentVec	: TEXCOORD3;	// position of viewer relative to point
  float2 mBumpCoord		: TEXCOORD4;
};

/*
** TEMP_OUT is a temporary structure for the ripple/Cyl Curl function
*/
struct TEMP_OUT
{
	float4 mPosition	: POSITION;
	float3 mNormal		: NORMAL0;
	float3 mTangent;
	float3 mBinormal;
};


/*
**------------------------------------------------
**		MakeRipple function
**------------------------------------------------
*/
TEMP_OUT MakeRipple( float3 position, uniform float param )
{
	TEMP_OUT returnVertex;
	
	float sinVal, cosVal, dist = 0;
	
	dist = distance ( float3(position.x*gPAR,position.yz), float3( gCentrePointX * gAspectRatio * gPAR, gCentrePointY, 0.1 ) );
	
	sincos( (dist+param)*10 , sinVal, cosVal );
	
    returnVertex.mPosition = float4(position.x,
									position.y, 
									position.z + param*0.25*cosVal,
									1.0f );
	
	returnVertex.mTangent = normalize( float3( 1.0, 0, param*0.4*sinVal*(position.x-gCentrePointX * gAspectRatio )/dist) );
	returnVertex.mBinormal= normalize( float3( 0, 1.0, param*0.4*sinVal*(position.y-gCentrePointY)/dist) );
	returnVertex.mNormal = cross( returnVertex.mTangent, returnVertex.mBinormal );
	
	return returnVertex;
}

/*
**-------------------------------------------------------------------------
** Ripple effect - Vertex Shader
**-------------------------------------------------------------------------
*/
VS_OUTPUT ripple_vs(APP_OUTPUT In)
{
    VS_OUTPUT Out = (VS_OUTPUT)0;
    
    In.mPosition.xy = gModelScaling*(In.mPosition.xy + gModelOffset.xy);
    In.mTexCoord.xy = gUVScaling*(In.mTexCoord.xy + gUVOffset.xy);
    
    // copy texture coordinates
	Out.mTexcoord.xy = In.mTexCoord.xy;
    Out.mBumpCoord.xy = Out.mTexcoord.xy;
    
    //Scale model to frame aspect ratio and send to MakeRipple
    TEMP_OUT tempVertex = MakeRipple( float3(In.mPosition.x*gAspectRatio,In.mPosition.yz), gCurrentParameter );
	
	// transform vertex position into homogenous clip-space
    Out.mHPosition = mul(gWorldViewProj, tempVertex.mPosition);
	
    // Generate BASIS matrix
	float3x3 rotationMat = { (tempVertex.mTangent), 
							  (tempVertex.mBinormal), 
							  (tempVertex.mNormal) };
	
	// store normalized light vector
    Out.mLightVec = (gLightVector.xyz - mul(gWorld,tempVertex.mPosition.xyz));
    float lightDist = distance( Out.mLightVec, float3(0.0,0.0,0.0) );
    Out.mLightVec /= lightDist;

	//compute the half vector    
    float4 eyePos = float4(0.0, 0.0, 1.8, 0.0);
    Out.mHalfVec = normalize(Out.mLightVec + eyePos);
    
    //compute Incident vector from eye position
    Out.mIncidentVec = normalize( mul(gWorld,tempVertex.mPosition.xyz) - eyePos);

	//bring the computed directions in s-t (texture) space	
	Out.mLightVec = mul( rotationMat, Out.mLightVec );
	Out.mHalfVec = mul(rotationMat, Out.mHalfVec );
	
	return Out;
}
 
/*
**-------------------------------------------------------------------------
** Ripple effect - pixel Shader 2.0 +
**-------------------------------------------------------------------------
*/
float4 ripple_ps_2_0(VS_OUTPUT In) : COLOR
{   
	float4 outColor, color1, reflectColor;
	float3 normal;
	float diffuse, specular;
	In.mLightVec = normalize( In.mLightVec );
	In.mHalfVec = normalize( In.mHalfVec );
	
	color1 = tex2D( Sampler, In.mTexcoord );
	
	//find out the bump normal
	normal = tex2D( SamplerBump, In.mBumpCoord ).xyz;
	normal = 2.0*(normal - 0.5);
	normal = lerp( float3( 0.0,0.0,1.0 ), normal, gBumpAmount );

	diffuse = dot(normal, In.mLightVec );
	specular = pow(dot(normal, In.mHalfVec ),32);
	color1.xyz = color1*(diffuse + specular);

	float3 R = reflect( In.mIncidentVec, normal );
	reflectColor = texCUBE( SamplerEnv, R );
	outColor.xyz = lerp( color1.xyz, reflectColor.xyz, gReflectivity );

	// two sampling on the noise texture to make noise look non-repeating.
	float3 fleckN      = (float3)tex2D(SamplerFleck, In.mTexcoord*19);
	fleckN      = fleckN + (float3)tex2D(SamplerFleck, In.mTexcoord*11) - 1;
	specular = saturate( dot(fleckN, In.mHalfVec ) );
	specular = pow ( specular, 64 );
	
	//put the specular color due to fleck
	outColor.xyz += gFleckAmount*specular*float3(0.6, 0.5, 0.1);
	outColor.a = color1.a;
	
    return outColor;
}

/*
**-------------------------------------------------------------------------
** Ripple effect - pixel Shader 1.4 
**-------------------------------------------------------------------------
*/
float4 ripple_ps_1_4(VS_OUTPUT In) : COLOR
{   
	float4 outColor, color1, reflectColor;
	float3 normal;
	float diffuse, specular;
	
	color1 = tex2D( Sampler, In.mTexcoord );
	
	//find out the bump normal
	normal = tex2D( SamplerBump, In.mBumpCoord ).xyz;
	normal = 2.0*(normal - 0.5);
	normal = lerp( float3( 0.0,0.0,1.0 ), normal, gBumpAmount );

	diffuse = dot(normal, In.mLightVec );
	specular = In.mHalfVec.z;
	color1.xyz = color1*(diffuse + specular*specular*0.33);

	float3 R = reflect( In.mIncidentVec, normal );
	reflectColor = texCUBE( SamplerEnv, R );
	outColor.xyz = lerp( color1.xyz, reflectColor.xyz, gReflectivity );

	outColor.a = color1.a;
	
    return outColor;
}


/*
**-------------------------------------------------------------------------
** Ripple effect - pixel Shader 1.3 
**-------------------------------------------------------------------------
*/
float4 ripple_ps_1_3(VS_OUTPUT In) : COLOR
{   
	float4 outColor, color1, reflectColor;
	float diffuse, specular;
	
	color1 = tex2D( Sampler, In.mTexcoord );
	
	diffuse = In.mLightVec.z;
	specular = In.mHalfVec.z;
	specular *= specular;
	specular *= specular;
	specular *= specular;
	specular *= specular;
	
	outColor.xyz = color1*(diffuse) + (specular)*color1*float3(0.6,0.6,0.6);
	outColor.a = color1.a;
	
    return outColor;
}
 
/*-----------------------------------------------------------------------------
**		technique(s) for various shader versionss
**-----------------------------------------------------------------------------
*/
technique ripple_2_0
{
    pass Pass0
    {
		Sampler[0] = (Sampler); 
		Sampler[1] = (SamplerBump);
		Sampler[2] = (SamplerEnv);
		Sampler[3] = (SamplerFleck);
		
		VertexShader = compile vs_1_1 ripple_vs();
		PixelShader  = compile ps_2_0 ripple_ps_2_0();
    }
}

technique ripple_1_4
{
    pass Pass0
    {
		Sampler[0] = (Sampler); 
		Sampler[1] = (SamplerBump);
		Sampler[2] = (SamplerEnv);
				
		VertexShader = compile vs_1_1 ripple_vs();
		PixelShader  = compile ps_1_4 ripple_ps_1_4();

        CullMode     = NONE;
	}
}

technique ripple_1_3
{
    pass Pass0
    {
		Sampler[0] = (Sampler); 
						
		VertexShader = compile vs_1_1 ripple_vs();
		PixelShader  = compile ps_1_3 ripple_ps_1_3();
    }
}




